class RagChain extends ReplicationInfo;

//////////////////////////////////
//
// Chain Links
//
//////////////////////////////////
//
// ChainPawn is the leader/owner of this chain.
// He pulls the first link, which pulls the second, ect..
//
var Pawn  ChainPawn;
var PlayerReplicationInfo PRI;
var Vehicle CurrentVehicle; //Gunna try to immitate PRI's CurrentVehicle with this.
//
// Ragdoll links in the chain
//
var array<xPawn> ChainLinks;
//
// Links in the chain that do not currently have ragdoll physics.
// Store them here temporary to give them time to change back to ragdoll physics.
//
var array<xPawn> BadLinks;

//////////////////////////////////
//
// Chain State Properties
//
//////////////////////////////////
//
// Server assigns to this property
// Clients only read from this property
//
// KillerChain == None .. I am a normally operating chain.
// KillerChain != None .. I have been destroyed/killed by KillerChain
//                        I should give my links + pawn to this KillerChain.
//                        I should destroy myself.
//
var RagChain KillerChain;
//
// Set to true after ChainPawn has been claimed by another chain.
// This is used to make sure that we do not append ChainPawn
// to the same chain multiple times
// (in the case of multiplayer, where this was happening)
//
var bool bAppended;
//
// Used by server when using timer and destroy
//
var int SDCount;

//////////////////////////////////
//
// Behavior Config Properties
//
//////////////////////////////////
//
//  This is the class responsible for
//  pulling the ragdolls.
//  This class determines the bahavior
//  of the ragdolls.
//  The default ChainPullerClass must
//  be set by the MutRagChain.
//  RagChain will use ChainPullerClass to
//  get a ChainPuller
//
var class<IChainPuller> ChainPullerClass;
var IChainPuller ChainPuller;
//
// The minimum count of ragdolls a chain must have
// before the chain starts deresing them.
//
var int MinRagdolls;
//
// How much distance is there between ragdoll links in the chain.
// Default at 30.
//
var float RagdollSeparationDistance;
//
// How long will a ragdoll take to deres.
// Set to -1 to use default.
//
var float RagdollLifeSpan;
//
// Set this to true to give radolls no collision
//
var bool bZeroCollisionRagdolls;
//
// Set this to true to give ragdolls zero gravity
//
var bool bZeroGravityRagdolls;

//////////////////////////////////
//
// Potential Behavior Config Properties (not currently configurable)
//
//////////////////////////////////
//
// Are the ragdolls in this chain bKImportantRagdoll==true?
// Currently only set in defaultproperties.
// Currently always set to true
//
var bool bKImportantRagdoll;
//
// (Single player & listen server only?)
// Should we bother pulling chains for pawns who are off screen?
// Maybe save some resources by turning this off.
// (unreal behaves like this is false for clients)
//
var bool bPullOffScreenChains;

//
//This is how the server tells clients that someone died.
//Also, the server needs to remind clients who's chain are for who.
//
replication
{
    reliable if(Role==ROLE_Authority && KillerChain != None)
        KillerChain;//Who killed this chain?
    reliable if(Role==ROLE_Authority && ChainPawn != None)
        ChainPawn;  //Who is this chain for?
    reliable if(Role==ROLE_Authority && PRI != None)
        PRI;        //Who is this chain for? (backup)
}

// This is just a log spam for testing
simulated function TestTick()
{
    local string pname;
    local Pawn P;
    local int count, driverMatch, chainPawnGood, PRIVehicleGood, priDriver, vdriverMatch, vdriverMatchWrong, vMatch;
    if(Level.NetMode != NM_Client || IsPlayersChain() || PRI == None) return;
    count = 0;
    driverMatch = 0;
    vdriverMatch = 0;
    vdriverMatchWrong = 0;
    vMatch = 0;

    if(ChainPawn == None)
    {
        chainPawnGood = 0;
    }
    else
    {

        if(ChainPawn.DrivenVehicle != None)
            chainPawnGood = 2;
        else
            chainPawnGood = 1;
    }

    //PRI.CurrentVehicle is usually None
    //Dur.. I'm dumb =B .. PRI.CurrentVehicle is not replicated.
    //Clients just use it as a cache reference sometimes for messages or something.
    /*
    if(PRI.CurrentVehicle == None)
    {
        if(ChainPawn == None)
        PRIVehicleGood = 0;
        else
        PRIVehicleGood = 1;
    }//These mostly never happens:
    else if(PRI.CurrentVehicle.PlayerReplicationInfo == PRI)
    {
        PRIVehicleGood = 2;
        if(PRI.CurrentVehicle.Driver != None)
        {
            if(PRI.CurrentVehicle.Driver == ChainPawn)
                priDriver = 2;
            else
                priDriver = 3;
        }
        else
                priDriver = 0;

        if(PRI.CurrentVehicle == ChainPawn.DrivenVehicle)
                vMatch = 2;
    }
    else
    {
        PRIVehicleGood = 3;
        if(PRI.CurrentVehicle.Driver != None)
        {
            if(PRI.CurrentVehicle.Driver == ChainPawn)
                priDriver = 4;
            else
                priDriver = 6;
        }
        else
                priDriver = 1;
    }
    */

    foreach AllActors(class'Pawn', P)
    {
        if(P.PlayerReplicationInfo == PRI)
        {
            ++count;
            if(P.PlayerReplicationInfo.CurrentVehicle != None && P.PlayerReplicationInfo.CurrentVehicle.Driver == ChainPawn)
                driverMatch++;
            if(Vehicle(P) != None)
            {
                if(Vehicle(P).Driver == ChainPawn)
                {
                    vdriverMatch++;
                }
                else
                {
                    vdriverMatchWrong++;
                }
            }
        }
    }
    Log("CP" @ chainPawnGood @ "PRIC" @ count @ "PRIDM" @ driverMatch @ "VPM" @ vdriverMatch @ "VPMW" @ vdriverMatchWrong);
}

simulated event Tick(float DeltaTime)
{
    //TestTick();

    // Clients will "forget" who ChainPawn is. (it may become "irrelevant")
    // The server has to reassign ChainPawn = ChainPawn
    // before unreal will replicate the value again. i guess bOnlyDirtyReplication == true
    // So we do this to force bNetDirty and "refresh"
    ChainPawn = ChainPawn;


    if(KillerChain != None)
    {
        if(KillerChain != Self)
        {   //KillerChain killed us, give him our ragdolls
            KillerChain.Append(Self);
        }
        else
        {   //we killed ourself
            ReleaseChain();
        }
        if(Level.NetMode == NM_DedicatedServer || Level.NetMode == NM_ListenServer)
            SetSelfDestruct();
        else
            Destroy();//clients will have to wait for the server to call Destroy()
    }
    else if(Level.NetMode != NM_DedicatedServer)
    {
        //Normal tick
        Tick_MaintainChain(DeltaTime);
        Tick_PullChain(DeltaTime);
    }
}

//
// The server will destroy itself after a delay.
// This is to give the clients time to get KillerChain and call Append.
//
function SetSelfDestruct()
{
    if(TimerRate == 0)
    {
        SetTimer(5, false);
    }
}
function Timer()
{
    Destroy();
}
simulated event Destroyed()
{
   ReleaseChain();
}

simulated function Tick_PullChain(float DeltaTime) //DeltaTime not currently used
{
    local Pawn CurrentChainHead;
    CurrentChainHead = GetChainHead();
    if(//Cannot pull this chaing
       CurrentChainHead == None)
    //|| (//don't need to pull this chain
           //!bPullOffScreenChains
           //!bZeroGravityRagdolls
        //&& !IsRelaventChain()))
    {
        return;
    }
    GetChainPuller().TickPull(DeltaTime,CurrentChainHead,ChainLinks);
}
simulated function bool IsRelaventChain()
{
    local Pawn ChainHead;
    local float lastRenderTimeMax;
    if(IsPlayersChain()) return true;

    ChainHead = GetChainHead();
    if(ChainHead != None)
    {
        lastRenderTimeMax = ChainHead.LastRenderTime;

        if(ChainPawn != None)
            lastRenderTimeMax = FMax(lastRenderTimeMax, ChainPawn.LastRenderTime);

        return Level.TimeSeconds - lastRenderTimeMax > 45;
    }

    return false;

}
simulated function bool IsPlayersChain()
{
    return ChainPawn != None
        && (  PlayerController(GetChainHead().Controller) != None
           || PlayerController(ChainPawn.Controller) != None);
}
simulated function IChainPuller GetChainPuller()
{
    if(ChainPuller==None)
    {
        ChainPuller = ChainPullerClass.static.NewChainPuller(self);
    }
    return ChainPuller;
}

simulated function Tick_MaintainChain(float DeltaTime) //DeltaTime is not used
{
    local xPawn CurrentLink;
    local int i;
    local bool bMovingChain;
    bMovingChain = ChainPawn != None;

    HandleBadChainLinks();

    for(i = 0; i < ChainLinks.Length; ++i)
    {
        CurrentLink = ChainLinks[i];
        if(!CurrentLink.bDeRes && bZeroCollisionRagdolls && CurrentLink.bCollideActors)
            CurrentLink.SetCollision(false,false);
        class'RagUtils'.static.ResetLastRenderTime(CurrentLink);
        //class'RagUtils'.static.SetKImportance(CurrentLink, bPullOffScreenChains || IsRelaventChain());
        // Reset the lifespan for every link but the last
        // unless you have less than MinRagdolls links.
        //
        if(i < ChainLinks.Length-1 || i < MinRagdolls)
        {
            if(RagdollLifeSpan > 0) //Use configged life span
                CurrentLink.LifeSpan = RagdollLifeSpan;
            else                    //Use default life span
                class'RagUtils'.static.ResetRagdollLifeSpan(CurrentLink);
        }

        // I think that calling this every tick was causing ragdolls
        // to telefrag players.
        // So I've taken it out.
        // Doesn't seem like it was necessary in the first place.
        //if(bMovingChain)
        //    class'RagUtils'.static.RagdollKWake(CurrentLink);
        //
    }
}

//Remove lost links, store bad ones, restore good ones
simulated function HandleBadChainLinks()
{
    local int i;
    /**
     *  If we got any pawns that are not currently "ragdolls"
     *  move them to the BadLinks for now.
     */
    for(i = 0; i < ChainLinks.Length; ++i)
    {
        if(ChainLinks[i] == None)
        {
            ChainLinks.Remove(i,1);
            --i;
        }
        else if(!class'RagUtils'.static.IsARagdoll(ChainLinks[i]))
        {
            BadLinks[BadLinks.Length] = ChainLinks[i];
            ChainLinks.Remove(i,1);
            --i;
        }
    }

    /**
     *  If a problem pawn becomes a ragdoll,
     *  put him back on the chain.
     */
    for(i = 0; i < BadLinks.Length; ++i)
    {
        if(BadLinks[i] == None)
        {
            BadLinks.Remove(i,1);
            --i;
        }
        else if(class'RagUtils'.static.IsARagdoll(BadLinks[i]))
        {
            ChainLinks[ChainLinks.Length] = BadLinks[i];
            BadLinks.Remove(i,1);
            --i;
        }
    }
}

//
// Take all of the pawns from the other chain.
//
simulated function Append(RagChain Other)
{
    local int i;
    local xPawn OtherChainPawn;

    if(Other.bAppended || Level.NetMode == NM_DedicatedServer)
        return; //Already claimed this chain, don't want duplicate references.
                //Dedicated servers do not care about ragdolls.


    OtherChainPawn = xPawn(Other.ChainPawn);
    if(OtherChainPawn != None)
    {
        if(!OtherChainPawn.bDeRes && bZeroGravityRagdolls)
            OtherChainPawn.KSetActorGravScale(0.0);
        //Log("Appending OtherChainPawn");
        Other.bAppended = AppendXPawn(xPawn(Other.ChainPawn));
    }
    else
    {
        //Log("Appending nothing, Other ChainPawn is none");
    }

    for(i = 0; i < Other.ChainLinks.Length; ++i)
        AppendXPawn(Other.ChainLinks[i]);

    for(i = 0; i < Other.BadLinks.Length; ++i)
        AppendXPawn(Other.BadLinks[i]);

    Other.ChainLinks.Length = 0;
    Other.BadLinks.Length = 0;
}
simulated function bool AppendXPawn(xPawn Other)
{
    if(class'RagUtils'.static.IsARagdoll(Other))
    {
        ChainLinks[ChainLinks.Length] = Other;
        class'RagUtils'.static.SetKImportance(Other, IsPlayersChain());
        return true;
    }
    else if(Other != None)
    {
        BadLinks[BadLinks.Length] = Other;
        class'RagUtils'.static.SetKImportance(Other, IsPlayersChain());
        return true;
    }
    else
    {
        //Log("NVM.  Append cancel.  Other chain pawn is not an xpawn");
        return false;
    }
}

simulated function Pawn GetChainHead()
{
    if(ChainPawn != None && ChainPawn.DrivenVehicle != None)
        return ChainPawn.DrivenVehicle;
    else if(ChainPawn != None)
        return ChainPawn;
    else if(PRI != None)// && PRI.CurrentVehicle != None && PRI.CurrentVehicle.PlayerReplicationInfo == PRI)
    {
        if(CurrentVehicle != None && CurrentVehicle.PlayerReplicationInfo == PRI)
            return CurrentVehicle;
        else if(PRI.CurrentVehicle != None && PRI.CurrentVehicle.PlayerReplicationInfo == PRI)
            return PRI.CurrentVehicle;
        else if ( Level.TimeSeconds - Level.LastVehicleCheck > 1 ) //From PRI.GetLocationName()
        {
            // vehicles are often bAlwaysRelevant, so may still be relevant
            ForEach DynamicActors(class'Vehicle', CurrentVehicle)
                if ( CurrentVehicle.PlayerReplicationInfo == PRI )
                {
                    if(Level.NetMode == NM_Client)
                        PRI.CurrentVehicle = CurrentVehicle;//Don't freakout, this is fine

                    return CurrentVehicle;
                }
            Level.LastVehicleCheck = Level.TimeSeconds;
        }
    }
    return None;
}

simulated function ReleaseChain()
{
    local int i;
    for(i = 0; i < ChainLinks.length; ++i)
        if(ChainLinks[i] != None)
            NormalizeRagdoll(ChainLinks[i], bZeroCollisionRagdolls);

    for(i = 0; i < BadLinks.Length; ++i)
        if(BadLinks[i] != None)
            NormalizeRagdoll(BadLinks[i], bZeroCollisionRagdolls);

    ChainLinks.Length = 0;
    BadLinks.Length = 0;
}
/*
simulated function UnstickRagdoll(Vector target, Pawn ragdoll)
{
    if(Distance(target,ragdoll.Location) > 3 * RagdollSeparationDistance)
    {
        ragdoll.SetLocation(target);
    }
}
*/


static function NormalizeRagdoll(xPawn Ragdoll, bool bZeroCollisionRagdolls)
{
    Ragdoll.LifeSpan = Fmin(Ragdoll.LifeSpan,Ragdoll.RagdollLifeSpan);
    class'RagUtils'.static.SetKImportance(Ragdoll,false);
    if(!ragdoll.bDeRes && bZeroCollisionRagdolls)
    {
        ragdoll.bWarping = true;//maybe avoid telefragging? (just guessing)
        ragdoll.SetCollision(true,false);
    }
}

defaultproperties
{
    RagdollSeparationDistance=30.00
    MinRagdolls = 3
    bPullOffScreenChains = true
    RagdollLifeSpan = -1.0
    bZeroCollisionRagdolls = false
    bZeroGravityRagdolls = false
    ChainPullerClass=class'SnakeChainPuller'
    ChainPuller=None

    KillerChain = None
    bAppended = false

    NetUpdateFrequency=3

    bKImportantRagdoll=True
}
