class CameraControl_CameraComponent extends ActorComponent;

`include(XS_Base/Classes/Library_Message.uci)

//------------------------------------------------------------------------------
// Required interface.
//------------------------------------------------------------------------------
////------------------------------------------------------------------------------
//// Instance vars.
////------------------------------------------------------------------------------
//
//var(Camera) CameraControl_CameraComponent Camera;
//var(Camera) config int CameraOffsetIndex;
//
////------------------------------------------------------------------------------
//// Events.
////------------------------------------------------------------------------------
//
//simulated event PostBeginPlay()
//{
//    super.PostBeginPlay();
//
//    Camera.SetVehicle(self);
//}
//
//simulated function VehicleCalcCamera(float DeltaTime, int SeatIndex, out vector CameraLocation, out rotator CameraRotation, out vector CameraStart, optional bool bPivotOnly)
//{
//    super.VehicleCalcCamera(DeltaTime, SeatIndex, CameraLocation, CameraRotation, CameraStart, bPivotOnly);
//
//    Camera.CalcCamera(CameraRotation);
//}
//
//// Note: It would appear that ANY seat in the vehicle can call this...
//simulated public function AdjustCameraScale(bool bMoveCameraIn)
//{
//    Camera.AdjustCamera(bMoveCameraIn);
//}
//
//simulated event DisplayHud(UTHUD Hud, Canvas Canvas, vector2D HudPosition, optional int SeatIndex)
//{
//	super.DisplayHud(Hud, Canvas, HudPosition, SeatIndex);
//
//	Camera.CheckFade(Canvas);
//}
//
////------------------------------------------------------------------------------
//// CameraControl_Interface.
////------------------------------------------------------------------------------
//
//function int GetCameraOffsetIndex()
//{
//    return CameraOffsetIndex;
//}
//
//function SetCameraOffsetIndex(int NewCameraOffsetIndex)
//{
//    CameraOffsetIndex = NewCameraOffsetIndex;
//}

//------------------------------------------------------------------------------
// Definitions.
//------------------------------------------------------------------------------

struct CameraInfo
{
	var bool bFirstPerson;
	var name FirstPersonBone;
	var Vector FirstPersonOffset;

	var float Height;
    var float Distance;
};

struct Position
{
	var Vector Location;
	var Rotator Rotation;
};

//------------------------------------------------------------------------------
// Instance vars.
//------------------------------------------------------------------------------

var int LastCameraOffsetIndex;

var(ThirdPerson) array<CameraInfo> Offsets;
var(ThirdPerson) float InterpolationSpeed;
var float InterpolationEndTime;

// How much of the craft's roll is transferred to the camera.  Note: Be careful;
// if the craft ends up on its back, the camera will alternate between left and
// right like crazy.
var(Camera) float Wobbliness;
// Speed factor multiplier for determining camera FOV.
var(Camera) float CameraFOVSpeedFactor;
// Speed factor exponent for determining camera FOV.
var(Camera) float CameraFOVSpeedFactorExponent;

var(Fade) float FadeDuration;
var float FadeEndTime;

// Reference to the vehicle using the camera interface.
var CameraControl_Interface MyCameraVehicle;
// Reference to the vehicle using the regular vehicle interface.
var UTVehicle MyVehicle;

//------------------------------------------------------------------------------
// Public functions (interface for the vehicle).
//------------------------------------------------------------------------------

public simulated function SetVehicle(UTVehicle NewVehicle)
{
    MyVehicle = NewVehicle;
    MyCameraVehicle = CameraControl_Interface(MyVehicle);

    UpdateCameraOffsetIndex(MyCameraVehicle.GetCameraOffsetIndex());
}

public simulated function CalcCamera(out vector CameraLocation, out rotator CameraRotation)
{
    if(GetDesiredOffset().bFirstPerson)
	{
	    CalcFirstPersonCamera(CameraLocation, CameraRotation);
	}
	else
	{
	    CalcThirdPersonCamera(CameraRotation);
	}
}

private simulated function CalcFirstPersonCamera(out vector CameraLocation, out rotator CameraRotation)
{
	local CameraInfo DesiredOffset;
	local Position FirstPersonPosition;

	DesiredOffset = GetDesiredOffset();

	if(DesiredOffset.FirstPersonBone != '')
	{
		FirstPersonPosition.Location = MyVehicle.Mesh.GetBoneLocation(DesiredOffset.FirstPersonBone);
		FirstPersonPosition.Rotation = Rotator(MyVehicle.Mesh.GetBoneAxis(DesiredOffset.FirstPersonBone, AXIS_X));
		FirstPersonPosition.Rotation.Roll = MyVehicle.Rotation.Roll;
	}
	else
	{
		FirstPersonPosition.Location = MyVehicle.Location;
		FirstPersonPosition.Rotation = MyVehicle.Rotation;
	}


	CameraLocation = FirstPersonPosition.Location + (DesiredOffset.FirstPersonOffset >> FirstPersonPosition.Rotation);
	CameraRotation.Roll = FirstPersonPosition.Rotation.Roll;
}

private simulated function CalcThirdPersonCamera(out rotator CameraRotation)
{
    CheckInterpolation();
    RollCamera(CameraRotation);
    SetCameraFOV();
}

// Camera: Adjust camera offset based on player input.
public simulated function AdjustCamera(bool bMoveCameraIn)
{
    local int OldCameraOffsetIndex;
    local int NewCameraOffsetIndex;

    OldCameraOffsetIndex = MyCameraVehicle.GetCameraOffsetIndex();

    if(bMoveCameraIn)
    {
        NewCameraOffsetIndex = OldCameraOffsetIndex - 1;
    }
    else
    {
        NewCameraOffsetIndex = OldCameraOffsetIndex + 1;
    }

	NewCameraOffsetIndex = Clamp(NewCameraOffsetIndex, 0, Offsets.length - 1);

    if(NewCameraOffsetIndex != OldCameraOffsetIndex)
    {
        MyCameraVehicle.SetCameraOffsetIndex(NewCameraOffsetIndex);
        MyVehicle.SaveConfig();

        UpdateCameraOffsetIndex(OldCameraOffsetIndex);

		if(Offsets[OldCameraOffsetIndex].bFirstPerson
		|| Offsets[NewCameraOffsetIndex].bFirstPerson)
		{
			// Fade this transition out.
			FadeEndTime = MyVehicle.WorldInfo.TimeSeconds + FadeDuration;
		}

		// Leaving first person?
		if(Offsets[OldCameraOffsetIndex].bFirstPerson
		&& !Offsets[NewCameraOffsetIndex].bFirstPerson)
		{
			// Instantly finish interpolation.  Otherwise interpolation will 
			// resume from the location where it left off, which can mean that 
			// the camera may still be zooming IN after leaving first person.
	        MyVehicle.SeatCameraScale = GetDesiredDistance();
	        MyVehicle.EyeHeight = GetDesiredHeight();
		}
    }
}

//------------------------------------------------------------------------------
// Helper functions.
//------------------------------------------------------------------------------

private simulated function UpdateCameraOffsetIndex(int OldCameraOffsetIndex)
{
    LastCameraOffsetIndex = OldCameraOffsetIndex;
    InterpolationEndTime = MyVehicle.WorldInfo.TimeSeconds + 1 / InterpolationSpeed;
}

private simulated function CameraInfo GetDesiredOffset()
{
    local int CameraOffsetIndex;

    CameraOffsetIndex = MyCameraVehicle.GetCameraOffsetIndex();

    return Offsets[CameraOffsetIndex];
}

//------------------------------------------------------------------------------
// Offset.
//------------------------------------------------------------------------------

private simulated function CheckInterpolation()
{
    if(InterpolationEndTime >= MyVehicle.WorldInfo.TimeSeconds)
    {
        InterpolateDistance();
        InterpolateHeight();
    }
}

// Interpolate to the desired distance.
private simulated function InterpolateDistance()
{
    local float DesiredDistance;
    local float InterpolatedDistance;
    local float InterpolationAlpha;

    DesiredDistance = GetDesiredDistance();

    if(MyVehicle.SeatCameraScale != DesiredDistance)
    {
        InterpolationAlpha = GetInterpolationAlpha();
        // Note: This is NOT linear interpolation since we're interpolating
        // from the CURRENT distance to the desired distance.
        InterpolatedDistance = Lerp(MyVehicle.SeatCameraScale, DesiredDistance, InterpolationAlpha);
        MyVehicle.SeatCameraScale = InterpolatedDistance;
    }
}

private simulated function float GetDesiredDistance()
{
    return GetDesiredOffset().Distance;
}

// Interpolate to the desired height.
private simulated function InterpolateHeight()
{
    local float DesiredHeight;
    local float InterpolatedHeight;
    local float InterpolationAlpha;
    //local int SeatIndex;

    DesiredHeight = GetDesiredHeight();

    if(MyVehicle.EyeHeight != DesiredHeight)
    {
        InterpolationAlpha = GetInterpolationAlpha();
        // Note: This is NOT linear interpolation since we're interpolating
        // from the CURRENT height to the desired height.
        InterpolatedHeight = Lerp(MyVehicle.EyeHeight, DesiredHeight, InterpolationAlpha);

        // Update driver seat.
        MyVehicle.EyeHeight = InterpolatedHeight;

        //// Update all passenger seats.
        //for(SeatIndex = 0; SeatIndex < MyVehicle.Seats.length; SeatIndex++)
        //{
        //    MyVehicle.Seats[SeatIndex].SeatPawn.EyeHeight = InterpolatedHeight;
        //}
    }
}

private simulated function float GetDesiredHeight()
{
    return GetDesiredOffset().Height * MyVehicle.BaseEyeHeight;
}

private simulated function float GetInterpolationAlpha()
{
    local float InterpolationTimeLeft;
    local float InterpolationAlpha;

    InterpolationTimeLeft = InterpolationEndTime - MyVehicle.WorldInfo.TimeSeconds;
    InterpolationAlpha = 1 - InterpolationTimeLeft * InterpolationSpeed;

    return InterpolationAlpha;
}

//------------------------------------------------------------------------------
// Roll.
//------------------------------------------------------------------------------

// How 'bout a lil' camera wobbliness?
private simulated function RollCamera(out rotator CameraRotation)
{
    if(MyVehicle.Health > 0)
    {
        CameraRotation.Roll = CameraRotation.Roll + MyVehicle.Rotation.Roll * Wobbliness;
    }
}

//------------------------------------------------------------------------------
// FOV.
//------------------------------------------------------------------------------

// FOV/distance magic.
private simulated function SetCameraFOV()
{
    local float ForwardSpeedFactor;
    local float FOV;

    if(PlayerController(MyVehicle.Controller) != none
    && UTVehicleWeapon(MyVehicle.Weapon).GetZoomedState() == ZST_NotZoomed)
    {
        ForwardSpeedFactor = (MyVehicle.Velocity << MyVehicle.Rotation).X / MyVehicle.MaxSpeed;
        ForwardSpeedFactor = ForwardSpeedFactor ** CameraFOVSpeedFactorExponent;
        FOV = MyVehicle.DefaultFOV + ForwardSpeedFactor * CameraFOVSpeedFactor;
        PlayerController(MyVehicle.Controller).DesiredFOV = FOV;
    }
}

//------------------------------------------------------------------------------
// Fade.
//------------------------------------------------------------------------------

public simulated function CheckFade(Canvas Canvas)
{
	if(MyVehicle.WorldInfo.TimeSeconds <= FadeEndTime)
	{
		DrawFade(Canvas);
	}
}

private simulated function DrawFade(Canvas Canvas)
{
    local float FadeProgress;

    FadeProgress = (FadeEndTime - MyVehicle.WorldInfo.TimeSeconds) / FadeDuration;

    Canvas.DrawColor.A = 255 * FadeProgress;
    Canvas.SetPos(0,0);
    Canvas.DrawTile(Texture2D 'EngineResources.Black', Canvas.SizeX, Canvas.SizeY, 0.0, 0.0, 16, 16);
}

//------------------------------------------------------------------------------
// Properties.
//------------------------------------------------------------------------------

defaultproperties
{
	// First move the vehicle, then move the camera.
	TickGroup=TG_PostAsyncWork

    Offsets(0)=(Height=1.0,Distance=1.0)
    InterpolationSpeed=0.1

    Wobbliness=0.0

    CameraFOVSpeedFactor=10
    CameraFOVSpeedFactorExponent=2

	FadeDuration=0.3
}